import java.util.*;
import javax.media.j3d.*;

/**
 * A simple collision detector class.  This responds to a collision 
 * event by printing a message with information about the type of 
 * collision event and the object involved.  This is a variation
 * of the CollisionDetector class that prints information about
 * the object that is associated with this behaviour rather than
 * the object that has been collided with. An example of its use
 * is given in the SimpleCollision2 class.
 * @author I.J.Palmer
 * @version 1.0
 * @see CollisionDetector
 * @see SimpleCollision2
 */
public class CollisionDetector2 extends Behavior {
    /** The shape that is being watched for collisions. */
    protected Shape3D collidingShape;
    /** The separate criteria that trigger this behaviour */
    protected WakeupCriterion[] theCriteria;
    /** The result of the 'OR' of the separate criteria */
    protected WakeupOr oredCriteria;

    /**
     * @param theShape Shape3D that is to be watched for collisions.
     * @param theBounds Bounds that define the active region for this behaviour
     */
    public CollisionDetector2(Shape3D theShape, Bounds theBounds) {
	collidingShape = theShape;
	setSchedulingBounds(theBounds);
    }

    /**
     * This sets up the criteria for triggering the behaviour.
     * It creates an entry, exit and movement trigger, OR's these
     * together and then sets the OR'ed criterion as the wake up
     * condition.
     */
    public void initialize() {
	theCriteria = new WakeupCriterion[3];
	WakeupOnCollisionEntry startsCollision = new WakeupOnCollisionEntry(collidingShape);
	WakeupOnCollisionExit endsCollision = new WakeupOnCollisionExit(collidingShape);
	WakeupOnCollisionMovement moveCollision = new WakeupOnCollisionMovement(collidingShape);
	theCriteria[0] = startsCollision;
	theCriteria[1] = endsCollision;
	theCriteria[2] = moveCollision;
	oredCriteria = new WakeupOr(theCriteria);
	wakeupOn(oredCriteria);
    }

    /**
     * This is where the work is done.
     * This identifies the type of collision (entry, exit or movement) and 
     * prints a message stating that an object has collided with this object.
     * The userData field of the shape associated with this collision detector #
     * is used to identify the object.  Finally, the wake up condition is set
     * to be the OR'ed criterion again.
     */
    public void processStimulus(Enumeration criteria) {
	while (criteria.hasMoreElements()) {
		WakeupCriterion theCriterion = (WakeupCriterion) criteria.nextElement();
		if (theCriterion instanceof WakeupOnCollisionEntry) {
	    		System.out.println("Collided with " + collidingShape.getUserData());
		}
		else if (theCriterion instanceof WakeupOnCollisionExit) {
	    		System.out.println("Stopped colliding with  " + collidingShape.getUserData());
		} else {
			System.out.println("Moved whilst colliding with " + collidingShape.getUserData());
		}
	}
	wakeupOn(oredCriteria);
    }
}
